<?php
/*
Plugin Name: Let's WP Plugin Control
Description: Enable or Disable a WordPress Plugin on Specific Pages
*/

/**
 * Controls the rules by which plugins are manipulated
 *
 * @param array $plugins List of active plugins
 * @return array Modified list of active plugins
 */
function lwp_plugin_control($plugins)
{
    $lwp_controlled_plugins = wp_cache_get('lwp_controlled_plugins', 'plugins');
    if ($lwp_controlled_plugins !== false) {
        return $lwp_controlled_plugins;
    }
    wp_cache_set('lwp_original_plugins', $plugins, 'plugins');

    // Enable plugins on certain URLs
    $enabling_rules = array(
        'some-gallery/some-gallery.php' => array(
            '/gallery/'
        )
    );
    // Disable plugins on certain URLs
    $disabling_rules = array(
        'some-lightbox/some-lightbox.php' => array(
            '/contact/$',
            '/wp-admin/post.php'
        )
    );

    // Run enables
    $plugins = array_unique(
        array_merge($plugins, lwp_plugins_affected_by($enabling_rules))
    );

    // Run disables
    $plugins = array_diff($plugins, lwp_plugins_affected_by($disabling_rules));

    wp_cache_set('lwp_controlled_plugins', $plugins, 'plugins');
    return $plugins;
}
add_filter('option_active_plugins', 'lwp_plugin_control');

/**
 * Scan for affected plugins to manipulate
 *
 * @param array $rules Plugin files paied with URL rules
 * @return array Affected plugins by their dir + file name
 */
function lwp_plugins_affected_by($rules)
{
    $affected = array();
    $current_path = add_query_arg(array());
    foreach ($rules as $plugin => $paths) {
        // If any of the paths match the current path
        $matches = array_filter(
            $paths,
            function ($path) use ($current_path) {
                return (
                    $path === $current_path ||
                    preg_match('%'.$path.'%', $current_path)
                );
            }
        );
        if (empty($matches)) {
            continue;
        }
        $affected[] = $plugin;
        add_filter('plugin_action_links_'.$plugin, 'lwp_add_action_links');
    }
    return $affected;
}

/**
 * Show a red message if a plugin is affected by the MU plugin
 * It explains why it's not possible to change it by hand
 *
 * @param array $links Links in the plugin row, like "Activate"
 * @return array Changed links to show in the plugin row
 */
function lwp_add_action_links($links)
{
    unset($links['activate'], $links['deactivate']);
    array_unshift($links, '<span style="color:red;">Controlled by MU!</span>');
    return $links;
}

/**
 * Prevent saving controlled plugin states by 3rd party plugins
 *
 * @param array $new_value New list of active plugins
 * @param array $old_value Old list of active plugins
 * @return array New list of active plugins
 */
function lwp_prevent_saving_plugins($new_value, $old_value)
{
    $lwp_controlled_plugins = wp_cache_get('lwp_controlled_plugins', 'plugins');
    $lwp_original_plugins = wp_cache_get('lwp_original_plugins', 'plugins');

    sort($old_value);
    sort($new_value);
    sort($lwp_controlled_plugins);

    if (($new_value === $old_value) &&
        ($old_value === $lwp_controlled_plugins)) {
        return $lwp_original_plugins;
    }
    return $new_value;
}

add_filter(
    'pre_update_option_active_plugins',
    'lwp_prevent_saving_plugins', 10, 2
);
