<?php
/*
Plugin Name: Internal Link Tooltip Preview
Description: In articles, show a sneak peek of links to other posts.
Version: 1.1
Author: Firsh from Let's WP
Author URI: https://letswp.io
License: GPL-3.0-or-later
License URI: https://www.gnu.org/licenses/gpl-3.0.html
*/
class LetsWP_InternalLinkTooltipPreview {
    const DEV = false;

    public function __construct() {
        add_filter('the_content', [$this, 'augment_intenal_links']);
        add_action('wp_enqueue_scripts', [$this, 'scripts_and_styles']);
    }

    public function augment_intenal_links($content) {
        if (is_front_page() || !is_single() || !in_the_loop() || !is_main_query()) {
            return $content;
        }

        return preg_replace_callback(
            '%<a href="(https://[^/]+/[^/]+/)">%sim',
            function ($matches) {
                $post_id = url_to_postid($matches[1]);
                if (0 === $post_id) {
                    return $matches[0];
                }

                $image = wp_get_attachment_image_src(get_post_thumbnail_id($post_id), 'medium');

                if (false === $image) {
                    return $matches[0];
                }

                remove_filter('the_content', [$this, 'augment_intenal_links']);
                $atts = [
                    'data-post-id' => esc_attr($post_id),
                    'data-post-title' => esc_attr(get_the_title($post_id)),
                    'data-post-excerpt' => esc_attr(get_the_excerpt($post_id)),
                    'data-post-image-src' => esc_attr($image[0]),
                    'data-post-image-w' => esc_attr($image[1]),
                    'data-post-image-h' => esc_attr($image[2]),
                    'class' => 'internal-link-tooltip-preview',
                ];
                add_filter('the_content', [$this, 'augment_intenal_links']);
                
                return str_replace(
                    '<a',
                    '<a '.implode(' ', array_map(
                        function ($v, $k) {
                            return sprintf('%s="%s"', $k, $v);
                        },
                        $atts,
                        array_keys($atts)
                    )),
                    $matches[0]
                );
            },
            $content
        );
    }

    public function scripts_and_styles() {
        if (false === $this::DEV) {
            wp_enqueue_style(
                'tooltipster-bundle-style',
                plugin_dir_url(__FILE__).'css/iltp.min.css'
            );
            wp_enqueue_script(
                'internal-link-tooltip-preview-js',
                plugin_dir_url(__FILE__).'js/iltp.min.js',
                ['jquery'],
                false,
                true
            );
        } else {
            wp_enqueue_style(
                'tooltipster-bundle-style',
                plugin_dir_url(__FILE__).'css/tooltipster.bundle.min.css'
            );
            wp_enqueue_script(
                'tooltipster-bundle-js',
                plugin_dir_url(__FILE__).'js/tooltipster.bundle.min.js',
                ['jquery'],
                false,
                true
            );
            wp_enqueue_script(
                'color-thief-js',
                plugin_dir_url(__FILE__).'js/color-thief.umd.js',
                [],
                false,
                true
            );
            wp_enqueue_style(
                'internal-link-tooltip-preview-style',
                plugin_dir_url(__FILE__).'css/internal-link-tooltip-preview.css'
            );
            wp_enqueue_script(
                'internal-link-tooltip-preview-js',
                plugin_dir_url(__FILE__).'js/internal-link-tooltip-preview.js',
                ['jquery', 'tooltipster-bundle-js', 'color-thief-js'],
                false,
                true
            );
        }
    }
}
new LetsWP_InternalLinkTooltipPreview();
